//$**************************************************************
//$            Copyright (C) 2011 by L-1 Identity Solutions.
//$
//$  Name:        $Workfile: salijag1.h $
//$  Author:      $Author: erics $
//$  Description: 
//$     Header defining externally available entry points for
//$     the SALI DLL.
//$**************************************************************

#ifndef SALIJAG1_H
#define SALIJAG1_H       1

/************************************************************************
** Error Codes
**
** The codes defined below may be returned to the caller of SALI
** functions.  By convention, the SALI functions return zero (the
** standard Windows symbol NO_ERROR) to indicate success.
**
** Error codes are separated into internal and user categories.
**
** Internal errors should never occur.  They may indicate a problem
** with the IEEE-1394 adapter or driver on the host, or there may
** be a problem communicating with the scanner, or they may indicate
** a scanner protocol problem.
**
** User errors are those which the caller may take some action to
** correct.  They may indicate an invalid usage of a SALI function,
** or they may inform the caller of some condition that requires
** further action.
**----------------------------------------------------------------------*/
/*
** Internal Errors
*/
/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

#define    SALI_INTERNAL_ERR        9000

#define    SALI_MEMORY_ALLOC_ERR    (SALI_INTERNAL_ERR + 1)
#define    SALI_ADAPTER_ACCESS_ERR  (SALI_INTERNAL_ERR + 2)
#define    SALI_MAPPING_ERR         (SALI_INTERNAL_ERR + 3)
#define    SALI_BUS_RESET_ERR       (SALI_INTERNAL_ERR + 4)
#define    SALI_NO_SCANNER          (SALI_INTERNAL_ERR + 5)
#define    SALI_WAIT_OBJ_ERR        (SALI_INTERNAL_ERR + 6)
#define    SALI_BAD_CHANNEL         (SALI_INTERNAL_ERR + 7)
#define    SALI_ISOCH_REQ_ERR       (SALI_INTERNAL_ERR + 8)
#define    SALI_VIDEO_RCV_ERR       (SALI_INTERNAL_ERR + 9)

#define    SALI_ASYNCH_READ_ERR     (SALI_INTERNAL_ERR + 10)
#define    SALI_ASYNCH_WRITE_ERR    (SALI_INTERNAL_ERR + 11)
#define    SALI_CSR_ADDRESS_ERR     (SALI_INTERNAL_ERR + 12)
#define    SALI_CSR_WRITE_ERR       (SALI_INTERNAL_ERR + 13)
#define    SALI_UNEXPECTED_RESPONSE (SALI_INTERNAL_ERR + 14)
#define    SALI_FAIL_RESPONSE_ERR   (SALI_INTERNAL_ERR + 15)
#define    SALI_RESPONSE_TIMEOUT    (SALI_INTERNAL_ERR + 16)
#define    SALI_BAD_PIPE            (SALI_INTERNAL_ERR + 17)
#define    SALI_XCTRL_ABORT         (SALI_INTERNAL_ERR + 18)

/*----------------------------------------------------------------------*/
/*
** User Errors
*/
/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

#define    SALI_USER_ERR            9100

#define    SALI_NOT_INITIALIZED     (SALI_USER_ERR + 1)
#define    SALI_BUFFER_TOO_BIG      (SALI_USER_ERR + 2)
#define    SALI_BUFFER_TOO_SMALL    (SALI_USER_ERR + 3)
#define    SALI_TOO_MANY_FRAMES     (SALI_USER_ERR + 4)
#define    SALI_VIDEO_STOPPED       (SALI_USER_ERR + 5)
#define    SALI_VIDEO_TIMEOUT       (SALI_USER_ERR + 6)
#define    SALI_TERMINATED_ERR      (SALI_USER_ERR + 7)
#define    SALI_BAD_SCANNER_INDEX   (SALI_USER_ERR + 8)
#define    SALI_BAD_PARAMETER       (SALI_USER_ERR + 9)
#define    SALI_SCN_UNAVAILABLE     (SALI_USER_ERR + 10)
#define    SALI_CANCELED            (SALI_USER_ERR + 11)

/************************************************************************/
/*
** Other Constants
*/
#define SALI_MAX_SCANNERS       8           // max number of scanners
/*----------------------------------------------------------------------*/
/*
** Transfer Types
**
** Indicate the relative position of a block of data within a large
** data transfer.  They must be supplied to SaliPut... functions, and
** are reported by SaliGet... functions.
*/
/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

#define SALI_FIRST_XFER             (1 << 0)
#define SALI_LAST_XFER              (1 << 1)

/************************************************************************/
/*
** SALI DLL Entry Point Descriptions
**
** All SALI functions return an error code, which will either be zero
** to indicate success (NO_ERROR defined in winerror.h) or one of the
** SALI error codes defined above.
*/
/*----------------------------------------------------------------------*/

#ifdef WIN32
	#ifdef SALI_EXPORTS
		#define SALI_API __declspec(dllexport)
	#else
		#define SALI_API __declspec(dllimport)
	#endif
#else                   //Linux
   #define SALI_API
   #define WINAPI
   #define PVOID     void *
   #define DWORD     unsigned long
   #define ULONG     unsigned long
   #define UINT      unsigned int
   #define PUINT     unsigned int *
   #define PULONG    unsigned long *
   #define UCHAR     unsigned char
   #define PSTR      char *
   #define PCHAR     char *
   #define TCHAR     char
   #define BOOL      Boolean
   #define BOOLEAN   Boolean
   #define _tcscmp   strcmp
   #define NOERROR   0L
   #define NO_ERROR  0L
   #define ERROR_SUCCESS 0L
   #define INFINITE  0xFFFFFFFF
#endif 

#ifdef __cplusplus
extern "C" {
#endif


// List of attached devices; parameter of SaliGetDeviceList_N
struct s_saliDevList
{
   unsigned int   libType;
   TCHAR          devName[255];
};

// List of attached devices; parameter of SaliGetDeviceList
typedef struct _DEVICE_LIST {
   char           deviceName[255];
} DEVICE_LIST, *PDEVICE_LIST;

/*----------------------------------------------------------------------*/
/*
** Management
*/
/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

#ifdef WIN32
SALI_API DWORD     SaliGetVerInfo
/*
** Get SALI version resource block.  This function copies the version
** information block of the SALI module to the input parameter location.
*/
(
VS_FIXEDFILEINFO* pVsffi			// pointer to version resource block
									// to load
);
#endif

SALI_API DWORD     SaliGetVersion
/*
** Get SALI version string.  This function returns a string representing
** the the version of SALI DLL currently loaded in the form 
** "major#.minor#.build#.release#".
*/
(
char *cpVersionString,				// pointer to string buffer to load
unsigned int uMaxChars				// maximum size of buffer
);

//---------------------------------------------
// Multi-Scanner Management Functions
//    Communicate with a maximum of 
//    SALI_MAX_SCANNERS attached devices
//---------------------------------------------
//
// Does not create a scanner block nor attempt any device 
// communication
SALI_API DWORD SaliInitialize_N(void);

// Returns a list of names of currently visible scanners
SALI_API DWORD SaliGetDeviceList_N(PUINT, struct s_saliDevList *);

// Open the device by name and return a handle to access it
SALI_API ULONG SaliOpenDevice_N(PSTR, UINT, PULONG);

// Returns a list of names of currently visible scanners
//    Assumes direct linkage to SALI library
SALI_API DWORD SaliGetDeviceList(PUINT, PDEVICE_LIST);

// Open the device by name and return a handle to access it
//    Assumes direct linkage to SALI library
SALI_API ULONG SaliOpenDevice(const PSTR, PULONG);

// Close the device handle
SALI_API DWORD SaliCloseDevice_N(ULONG);

#ifdef WIN32
SALI_API DWORD SaliGetVerInfo_N(ULONG, VS_FIXEDFILEINFO*);
#endif
SALI_API DWORD SaliGetVersion_N(ULONG,char *,unsigned int);

SALI_API DWORD SaliBusReset_N(ULONG);
SALI_API DWORD SaliSetTimeouts_N(ULONG,DWORD,DWORD);

//---------------------------------------------
// Single-Scanner Management Functions
//---------------------------------------------
//
SALI_API DWORD     SaliInitialize
/*
** Initializes SALI communications.  This function opens the default
** IEEE-1394 adapter, which should be the only IEEE-1394 adapther in
** the host system.   It is an active initialization in that it will
** issue an IEEE-1394 bus reset and attempt to communicate with a
** scanner.  If this function succeeds then the caller will know that
** a scanner is available.
**
** This function allocates all of the memory it requires for scanner
** communications, except for isochronous, real-time image data.  In
** consequence, the calling process owns this memory, and should free
** it by calling SaliTerminate when finished.
**
** This function will wait up to 30 seconds for the IEEE-1394 driver
** to become available, and up to 30 seconds more for the adapter to
** respond to local access.  A non-zero "extraWait" argument causes
** this function also to wait the specified length of time at five
** other stages during initialization.  The units are milliseconds.
**
** The "scnCnt" pointer reports the number of scanner in the protocol.
*/
(
DWORD               extraWait,      /*<- extra time (msec)              */
unsigned            *scnCnt         /*-> scanner Count                  */
);

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

SALI_API DWORD   SaliGetBufSize
/*
** Gets the maximum sizes of various types of buffers that may be
** transferred.  
** 
** The "cmdBufSize" pointer reports the maximum size of a command
** message that can be sent with the SaliPutCommand function. 
** The "statBufSize" pointer reports the maximum size status message 
** that can be obtained by calling the SaliGetStatus function. 
** The "datBufSize" pointer reports the maximum size data block that 
** can be transferred with the bulk data transfer functions.
*/
(
unsigned            scnIndex,       /*<- scanner index, start from 0    */
size_t              *cmdBufSize,    /*-> command max size each scanner  */
size_t              *statBufSize,   /*-> status max size each scanner   */
size_t              *datBufSize     /*->other data max size each scanner*/
);

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

SALI_API DWORD     SaliTerminate
/*
** Ceases all IEEE-1394 communications and releases all allocated
** memory.
**
** If SALI is receiving real-time image frames, the caller should
** call SaliStopVideo before calling this function.
*/
(
 void
);

/*----------------------------------------------------------------------*/
/*
** Command Transfers
*/
/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

SALI_API DWORD     SaliPutCommand
/*
** Sends a command to a scanner.
**
** The "buffer" argument points to a scanner command.  The "bufLength"
** argument should specify the number of bytes in the command.  It must
** be no larger than the maximum command size reported by SaliInitialize.
**
** This function reports the actual number of bytes transferred using
** the "bufXferCnt" pointer argument.
**
** The scanner command messages and their structures are defined in
** the SCNPRTCL.H header file.
*/
(
unsigned            scnIndex,       /*<- scanner index, start from 0    */
void                *buffer,        /*<- pointer to command buffer      */
size_t              bufLength,      /*<- length of command              */
size_t              *bufXferCnt     /*-> bytes transferred              */
);

/*----------------------------------------------------------------------*/
/*
** Status Transfers
*/
/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

SALI_API DWORD     SaliGetStatus
/*
** Awaits a status message from a scanner.  Normally, this function
** does not return until it receives a new status message.  However,
** calling SaliTerminate will also force it to return.
**
** This function copies the received status message into the buffer
** pointed to by the "buffer" argument.  The "bufLength" argument
** specifies the number of bytes in the buffer.  It should be the
** maximum status size reported by SaliInitialize.
**
** This function reports the actual number of bytes transferred using
** the "bufXferCnt" pointer argument.  This may be greater than the
** actual size of the status message.  The size of the status message
** depends on its status code.
**
** The scanner status messages and their structures are defined in
** the SCNPRTCL.H header file.
*/
(
unsigned            scnIndex,           /*<- scanner index, start from 0*/
void                *buffer,            /*<- pointer to status buffer   */
size_t              bufLength,          /*<- length of buffer           */
size_t              *bufXferCnt         /*-> bytes transferred          */
);

/*----------------------------------------------------------------------*/
/*
** Bulk Data Transfers
**
** These transfers only occur in conjunction with certain commands.
** The caller must determine when to call these functions depending
** on which commands it issues and the status messages it receives.
**
** Bulk data transfers encompass image transfers, bitmap transfers,
** and memory transfers.  There are specifically named functions
** that provide for bulk data transfers in each of these categories.
** Their names were appropriate at an earlier time, when the protocol
** used to communicate with the scanner really did distinquish these
** categories of data transfers.  Now, the protocol treats all of
** these the same.  As an internal matter, all data transfers are
** now directed through the image transfer functions.
**
** The data buffer size, obtained via SaliGetBufSize, must be used
** as the maximum buffer size for these transfers.  If a transfer
** larger than this size is required, the transfer must start with
** one or more transfers of this full buffer size and end with a final
** transfer of the remaining size.  The exact use of the buffer size
** is important.
*/
/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

SALI_API DWORD     SaliPutImage
/*
** Transfers the supplied image buffer content to a scanner.
**
** The "buffer" argument points to a portion of the data.  The
** "bufLength" argument specifies how many bytes are in that block.
** It should be no more than the maximum block size reported by
** SaliInitialize.
**
** This function reports the actual number of bytes transferred using
** the "bufXferCnt" pointer argument.  It should equal the "bufLength"
** argument.
**
** The "xferType" argument indicates the location of the current block
** within the image.  If it is the first block then its value should be
** SALI_FIRST_XFER.  If it is the last block then its value should be
** SALI_LAST_XFER.  For an intermediate block it should be zero.
** Finally, if this is the only block then the "xferType" value should
** be (SALI_FIRST_XFER | SALI_LAST_XFER).
*/
(
unsigned            scnIndex,           /*<- scanner index, start from 0*/
void                *buffer,            /*<- pointer to image buffer    */
size_t              bufLength,          /*<- amount of image data       */
size_t              *bufXferCnt,        /*-> bytes transferred          */
DWORD               xferType            /*<- first, last, intermediate  */
);

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

SALI_API DWORD     SaliGetImage
/*
** Transfers the expected image buffer content from a scanner.
**
** The "buffer" argument points to a place in which to store the
** next portion of the data.  The "bufLength" argument specifies
** the size of the supplied buffer.  It should be the maximum size
** reported by SaliInitialize.
**
** This function reports the actual number of bytes transferred using
** the "bufXferCnt" pointer argument.
**
** This function reports the location of the current block within the
** image using the "xferType" pointer argument.  If it is the first
** block then the reported value will be SALI_FIRST_XFER.  If it is
** the last block then the reported value will be SALI_LAST_XFER.  If
** it is an intermediate block then the reported value will be zero.
** Finally, if it is the only block then the reported value will be
** (SALI_FIRST_XFER | SALI_LAST_XFER).
*/
(
unsigned            scnIndex,           /*<- scanner index, start from 0*/
void                *buffer,            /*<- pointer to image buffer    */
size_t              bufLength,          /*<- expected amount of data    */
size_t              *bufXferCnt,        /*-> bytes transferred          */
DWORD               *xferType           /*-> first, last, intermediate  */
);

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

SALI_API DWORD     SaliPutBitmap
/*
** Transfers the supplied bit-map buffer content to a scanner.
**
** See SaliPutImage above for details.
*/
(
unsigned            scnIndex,           /*<- scanner index, start from 0*/
void                *buffer,            /*<- pointer to bit-map buffer  */
size_t              bufLength,          /*<- amount of bit-map data     */
size_t              *bufXferCnt,        /*-> bytes transferred          */
DWORD               xferType            /*<- first, last, intermediate  */
);

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

SALI_API DWORD     SaliPutMemory
/*
** Transfers the supplied data buffer content to a scanner.
**
** See SaliPutImage above for details.
*/
(
unsigned            scnIndex,           /*<- scanner index, start from 0*/
void                *buffer,            /*<- pointer to data buffer     */
size_t              bufLength,          /*<- amound of other data       */
size_t              *bufXferCnt,        /*-> bytes transferred          */
DWORD               xferType            /*<- first, last, intermediate  */
);

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

SALI_API DWORD     SaliGetMemory
/*
** Transfers the expected data buffer content from a scanner.
**
** See SaliGetImage above for details.
*/
(
unsigned            scnIndex,           /*<- scanner index, start from 0*/
void                *buffer,            /*<- pointer to data buffer     */
size_t              bufLength,          /*<- expected amount of data    */
size_t              *bufXferCnt,        /*-> bytes transferred          */
DWORD               *xferType           /*-> first, last, intermediate  */
);

/*----------------------------------------------------------------------*/
/*
** Real-time Images (Video)
**
** These functions manage the reception of real-time image data for
** display on the host system.  During a capture process, the scanner
** sends real-time image data over an IEEE-1394 isochronous channel
** with its remote display feature.
**
** Each frame has the caller specified height and width.  The pixel
** data are sent from top to bottom and left to right.  Thus, the first
** byte in each frame buffer is the pixel in the upper-left corner of
** the image.  Band buffers follow the same orientation, but there can
** be several bands in each frame.
**
** The scheme for receiving real-time images is:
**
** 1) Call SaliStartVideo to indicate the image dimensions and to
**    prepare for receiving isochronous data.  This also reports the
**    size required for each buffer.  Call SaliSetVideoBandParams
**    first if a banded display is desired.
**
** 2) Allocate a set of buffers of the required size.  Submit all of
**    them by calling SaliSubmitVideoRequest once with each buffer
**    pointer.
**
** 3) Enter a loop to process real-time image data as indicated below:
**
**    a) Call SaliGetVideoResult to wait for a filled buffer.
**
**    b) If SaliGetVideoResult returned any error code, exit the loop.
**
**    c) Upon a successful return from SaliGetVideoResult, display the
**       contents of the reported buffer.  Ideally, this should be a
**       quick, synchronous operation, after which the buffer would be
**       available for re-use.  Call SaliGetVideoBandPosition to find
**       the vertical location to display the buffer when using banded
**       display.
**
**    d) When the buffer is available, re-submit the it by calling
**       SaliSubmitVideoRequest.
**
**    e) Go back to the top of the loop.
**
** 4) Call SaliStopVideo after exiting the loop.  This should be done
**    regardless of the reason for stopping.
**
** 5) Call SaliGetVideoResult repeatedly, until it reports a NULL
**    buffer pointer.  This retracts all previously submitted buffers.
**
** Call the SaliStopVideo function to stop video processing.  This will
** cause SaliGetVideoResult to return an error code (SALI_VIDEO_STOPPED),
** at which time the above procedure will retract all submitted buffers
** and cease attempting to receive real-time image frames.  Note that
** a separate thread is needed to call SaliStopVideo.
*/
#define SALI_VIDEO_UNBANDED        0     // Full image
/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

SALI_API DWORD      SaliStartVideo
/*
** Prepares SALI to receive real-time image frames from a scanner over
** an IEEE-1394 isochronous channel.
**
** The height of these frames is given by the "rows" argument and their
** width is given by the "columns" argument.  The "pitch" argument gives
** the amount of storage needed for each row; it must be the same as
** "columns".
**
** Because isochronous data are transferred in fixed-size packets, the
** actual amount of storage needed for each frame must be a multiple of
** the packet size.  This function reports the minimum size that the
** caller must allocate for each frame buffer using the "bufferSize"
** pointer argument.  It will be the smallest multiple of the packet
** size that is not less than the product of "rows" and "pitch".
**
**   NOTE:  The size given to this function must match the configured
**          remote image size in the scanner.
*/
(
unsigned            scnIndex,           /*<- scanner index, start from 0*/
size_t              rows,               /*<- rows of video data         */
size_t              columns,            /*<- columns of video data      */
size_t              pitch,              /*<- byte distance between rows */
size_t              *bufferSize         /*-> minimum size for buffers   */
);

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

SALI_API DWORD      SaliStopVideo
/*
** Causes SALI to stop receiving real-time image frames.  The caller
** should then repeatedly call SaliGetVideoResult to retract all
** previously submitted buffers.
**
** Subsequent calls to SaliSubmitVideoRequest and SaliGetVideoResult
** will return SALI_VIDEO_STOPPED.
*/
(
unsigned            scnIndex            /*<- scanner index, start from 0*/
);

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

SALI_API DWORD      SaliSubmitVideoRequest
/*
** Submits a buffer to be filled by a real-time image received from
** a scanner.  This function returns immediately.  Provided there was
** no error, the submitted buffer remains pending until it is reported
** by a subsequent call to SaliGetVideoResult.
**
** The "buffer" argument points to a frame buffer allocated by the
** caller.  It must be at least the minimum size reported by the prior
** call to SaliStartVideo.
**
** This function stores the "reference" argument.  SALI makes no direct
** use of this pointer, but SaliGetVideoResult will report it when its
** associated frame buffer is ready.  It may be NULL.
*/
(
unsigned            scnIndex,           /*<- scanner index, start from 0*/
void                *buffer,            /*<- video data buffer          */
void                *reference          /*<- caller's reference         */
);

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

SALI_API DWORD      SaliGetVideoResult
/*
** Returns when a previously submitted frame buffer is ready, and
** reports which frame buffer contains the next real-time image frame.
** This function also reports the reference pointer submitted along
** with that frame buffer.  It blocks the caller's thread until a
** frame buffer is ready, or another thread calls SaliStopVideo, or a
** catastrophic error occurs.
**
** The pointer to a previously submitted frame buffer is reported
** using the "buffer" pointer argument.  This pointer must point to
** a valid pointer variable because this function always reports a
** buffer, even when an error occurs.
**
** The reference pointer submitted with the frame buffer is reported
** using the "reference" pointer argument.  It may be NULL if the
** caller does not need the information.
*/
(
unsigned            scnIndex,           /*<- scanner index, start from 0*/
void                **buffer,           /*-> caller's original buffer   */
void                **reference         /*-> caller's reference         */
);

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

SALI_API DWORD      SaliSetVideoBandParams
/*
** Establishes whether to expect full video frames or frames divided
** into bands.  A value of zero selects full frame mode.  A non-zero
** value selects band mode and sets the number of lines per band.  The
** second argument sets the initial time-out when waiting for the first
** band.
**
** The specified band size will not take effect until the next call
** to SaliStartVideo.
**
**   NOTE:  The band size given to this function must match the
**          configured remote image band size in the scanner.
*/
(
unsigned            scnIndex,           /*<- scanner index, start from 0*/
size_t              lines,              /*<- lines per band             */
unsigned long       timeout             /*<- initial band timeout (ms)  */
);

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

SALI_API DWORD      SaliGetVideoBandPosition
/*
** Reports the vertical position of the band buffer most recently
** returned by SaliGetVideoResult, as a zero-origin line number.  It
** also reports a pointer within that buffer for the beginning of
** valid video data and the number of valid lines in the band buffer.
*/
(
unsigned            scnIndex,           /*<- scanner index, start from 0*/
size_t              *position,          /*-> line pos of band buffer    */
void                **data,             /*-> actual beginning of data   */
size_t              *lines              /*-> number of valid lines      */
);

/*----------------------------------------------------------------------*/

/*----------------------------------------------------------------------*/
/*
** Entry Points for Debug, etc.
*/

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

SALI_API DWORD     SaliBusReset
/*
** This is an experimental entry point.  It may prove to have no
** application value.  It sends a bus reset.
**
** Until this comment is removed, this function should only be
** used in test programs.
*/
(
void
);

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

SALI_API DWORD     SaliSetTimeouts
/*
** Override the default timeout values for SALI.
*/
(
DWORD               asynchPutTimeout,
DWORD               asynchGetTimeout
);

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -*/

#ifdef WIN32
SALI_API void      SaliBreakEnable
(
BOOL               enableIfTrue
);
#endif

/*----------------------------------------------------------------------*/

#ifdef __cplusplus
}
#endif

#endif  /* #ifdef SALIJAG1_H */

/************************************************************************/
/* end of salijag1.h */
